<?php

namespace App\DataTables;

use App\Models\Coupons;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Illuminate\Database\Query\JoinClause;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Services\DataTable;

class CouponsDataTable extends DataTable
{
    const STATUS_NEW = 0;

    const STATUS_USED = 1;

    const STATUS_EXPIRED = 2;

    /**
     * Build the DataTable class.
     *
     * @param QueryBuilder $query Results from query() method.
     */
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        return (new EloquentDataTable($query))
            ->rawColumns(['status'])
            ->editColumn('exp', function (Coupons $coupons) {
                return $coupons->expired_date->format('d-m-Y');
            })
            ->editColumn('status', function (Coupons $coupons) {
                if ($coupons->status == self::STATUS_NEW) {
                    $statusText = 'Baru';
                    $statusColor = 'success';
                } else if ($coupons->status == self::STATUS_USED) {
                    $statusText = 'Terpakai';
                    $statusColor = 'warning';
                } else {
                    $statusText = 'Kadaluarsa';
                    $statusColor = 'danger';
                }
                return "<span class='badge badge-light-$statusColor fs-7 m-1'>$statusText</span>";
            })
            ->editColumn('action', function (Coupons $coupons) {
                return view('pages.apps.wheel.coupons.columns._actions', compact('coupons'));
            })
            ->setRowId('id');
    }

    /**
     * Get the query source of dataTable.
     */
    public function query(Coupons $model): QueryBuilder
    {
        return $model->newQuery()
            ->select('coupons.*', 'coupons.id AS coupon_id', 'prizes.text')
            ->leftJoin('prizes', function (JoinClause $join) {
                $join->on('coupons.prize_id', '=', 'prizes.id');
            });
    }

    /**
     * Optional method if you want to use the html builder.
     */
    public function html(): HtmlBuilder
    {
        return $this->builder()
            ->setTableId('coupons-table')
            ->columns($this->getColumns())
            ->minifiedAjax()
            ->dom('rt' . "<'row'<'col-sm-12 col-md-5'l><'col-sm-12 col-md-7'p>>",)
            ->addTableClass('table align-middle table-row-dashed fs-6 gy-5 dataTable no-footer text-gray-600 fw-semibold')
            ->setTableHeadClass('text-start text-muted fw-bold fs-7 text-uppercase gs-0')
            ->orders([[4, 'asc'], [0, 'desc']])
            ->pageLength(50)
            ->lengthMenu([50, 100, 200])
            ->drawCallback("function() {" . file_get_contents(resource_path('views/pages/apps/wheel/coupons/columns/_draw-scripts.js')) . "}");
    }

    /**
     * Get the dataTable columns definition.
     */
    public function getColumns(): array
    {
        return [
            Column::make('created_at')->searchable(false)->visible(false),
            Column::make('id'),
            Column::make('username'),
            Column::make('text')->title('PRIZE')->name('prizes.text'),
            Column::make('status')->searchable(false),
            Column::make('exp')->name('expired_date'),
            Column::computed('action')
                ->addClass('text-end text-nowrap')
                ->exportable(false)
                ->printable(false)
                ->width(60)
        ];
    }

    /**
     * Get the filename for export.
     */
    protected function filename(): string
    {
        return 'Coupons_' . date('YmdHis');
    }
}
