<?php

namespace App\DataTables;

use App\Models\SpinLog;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Illuminate\Database\Query\JoinClause;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;

class SpinLogDataTable extends DataTable
{
    /**
     * Build the DataTable class.
     *
     * @param QueryBuilder $query Results from query() method.
     */
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        return (new EloquentDataTable($query))
            ->rawColumns(['coupon_id'])
            ->editColumn('username', function (SpinLog $log) {
                return $log->username ?? '-';
            })
            ->editColumn('text', function (SpinLog $log) {
                return $log->text ?? '-';
            })
            ->editColumn('coupon_id', function (SpinLog $log) {
                $coupon = $log->coupon_id;
                return "<a target='_blank' href='" . asset("assets/screenshot/$coupon.webp") . "'>$coupon</a>";
            })
            ->editColumn('created_at', function (SpinLog $log) {
                return $log->created_at->format('d-m-Y');
            })
            ->setRowId('id');
    }

    /**
     * Get the query source of dataTable.
     */
    public function query(SpinLog $model): QueryBuilder
    {
        return $model->newQuery()
            ->select('history.*', 'coupons.username', 'coupons.id', 'prizes.text')
            ->leftJoin('coupons', function (JoinClause $join) {
                $join->on('history.coupon_id', '=', 'coupons.id');
            })
            ->leftJoin('prizes', function (JoinClause $join) {
                $join->on('coupons.prize_id', '=', 'prizes.id');
            });
    }

    /**
     * Optional method if you want to use the html builder.
     */
    public function html(): HtmlBuilder
    {
        return $this->builder()
            ->setTableId('logs-table')
            ->columns($this->getColumns())
            ->minifiedAjax()
            ->dom('rt' . "<'row'<'col-sm-12 col-md-5'l><'col-sm-12 col-md-7'p>>",)
            ->addTableClass('table align-middle table-row-dashed fs-6 gy-5 dataTable no-footer text-gray-600 fw-semibold')
            ->setTableHeadClass('text-start text-muted fw-bold fs-7 text-uppercase gs-0')
            ->orderBy(0)
            ->pageLength(50)
            ->lengthMenu([50, 100, 200, 300]);
    }

    /**
     * Get the dataTable columns definition.
     */
    public function getColumns(): array
    {
        return [
            Column::make('username', 'coupons.username'),
            Column::make('text', 'prizes.text')->title('hadiah'),
            Column::make('coupon_id')->title('kode kupon'),
            Column::make('ip_address')->title('ip address'),
            Column::make('platform'),
            Column::make('browser'),
            Column::make('mobile'),
            Column::make('referrer'),
            Column::make('created_at')->title('Tanggal Spin')->searchable(false),

        ];
    }

    /**
     * Get the filename for export.
     */
    protected function filename(): string
    {
        return 'SpinLog_' . date('YmdHis');
    }
}
