<?php

namespace App\Livewire;

use App\Livewire\Traits\ExceptionHandler;
use App\Models\Coupons;
use App\Models\Prizes;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Livewire\Attributes\Computed;
use Livewire\Component;

class BatchCoupon extends Component
{
    use ExceptionHandler;

    /**
     * Give context to custom exception handler
     *
     * @var array $context
     */
    protected $context = 'kupon';

    public $emptyCoupon = [
        'username' => null,
        'prize' => null,
        'expired' => null,
    ];

    public $coupons = [];

    public function messages()
    {
        return [
            'required' => ':attribute tidak boleh kosong',
            'string' => ':attribute tidak boleh memiliki karakter lain selain huruf',
            'numeric' => ':attribute hanya menerima angka',
            'min' => ':attribute harus lebih dari 0'
        ];
    }

    public function rules()
    {
        return [
            'coupons.*.username' =>  [
                'required'
            ],
            'coupons.*.prize' =>  [
                'required'
            ],
            'coupons.*.expired' =>  [
                'required',
                'date',
                function ($attribute, $value, $fail) {
                    $date = \Carbon\Carbon::parse($value);
                    $yesterday = \Carbon\Carbon::yesterday();

                    if ($date->lessThanOrEqualTo($yesterday)) {
                        $fail('Tanggal kadaluarsa harus lebih besar dari kemarin.');
                    }
                },
            ],
        ];
    }

    public function validationAttributes()
    {
        return [
            'coupons.*.username' => 'Username',
            'coupons.*.prize' => 'Tanggal kadaluarsa',
            'coupons.*.expired' => 'Hadiah',
        ];
    }

    #[Computed(persist: true)]
    public function prizes()
    {
        return Prizes::all(['id', 'text']);
    }

    /**
     * Mount livewire component
     *
     * @param string $type contain enum value between isDeposit | isWithdraw.
     *
     */
    public function mount(string $type = null)
    {
        $this->{$type} = true;
        $this->coupons[] = $this->emptyCoupon;
    }

    public function store()
    {
        $request = $this->validate();

        try {
            DB::transaction(function () use ($request) {
                foreach ($request['coupons'] as $item) {
                    Coupons::create([
                        'id' => strtoupper(Str::random(16, 'ABCDEFGHIJKLMNOPQRSTUVWXY1234567890')),
                        'prize_id' => $item['prize'],
                        'username' => $item['username'],
                        'expired_date' => $item['expired']
                    ]);
                }
            });
        } catch (\PDOException $e) {
            return $this->handleDatabaseException($e);
        } catch (\Exception $e) {
            return $this->handleDefaultException($e);
        }

        $this->dispatch('success', 'Berhasil menambahkan kupon');
        $this->reset();
    }

    public function hydrate()
    {
        $this->resetErrorBag();
        $this->resetValidation();
    }
}
