<?php

namespace App\Livewire;

use App\Livewire\Traits\ExceptionHandler;
use App\Models\Coupons;
use App\Models\Prizes;
use App\Models\SpinLog;
use Illuminate\Support\Facades\DB;
use Livewire\Attributes\On;
use Livewire\Attributes\Validate;
use Livewire\Component;
use Illuminate\Support\Str;
use Livewire\Attributes\Computed;

class CouponModal extends Component
{
    use ExceptionHandler;

    const STATUS_NEW = 0;

    const STATUS_USED = 1;

    const STATUS_EXPIRED = 2;

    /**
     * Give context to custom exception handler
     *
     * @var array $context
     */
    protected $context = 'kupon';

    public Coupons $coupon;

    public $title = 'Tambah Kupon';

    public $is_edit = false;

    #[Validate('required|string', as: 'Username', translate: false)]
    public $username;

    #[Validate(as: 'Tanggal kadaluarsa', translate: false)]
    public $expired;

    #[Validate('required', as: 'Hadiah', translate: false)]
    public $prize;

    public function rules()
    {
        return [
            'expired' => [
                'required',
                'date',
                function ($attribute, $value, $fail) {
                    $date = \Carbon\Carbon::parse($value);
                    $yesterday = \Carbon\Carbon::yesterday();

                    if ($date->lessThanOrEqualTo($yesterday)) {
                        $fail('Tanggal kadaluarsa harus lebih besar dari kemarin.');
                    }
                },
            ]
        ];
    }

    public function messages()
    {
        return [
            'required' => ':attribute tidak boleh kosong',
            'string' => ':attribute tidak boleh memiliki karakter lain selain huruf',
        ];
    }

    #[Computed(persist: true)]
    public function prizes()
    {
        return Prizes::all(['id', 'text']);
    }

    #[On('create')]
    public function create()
    {
        $this->reset();
    }

    public function store()
    {
        $request = $this->validate();

        try {
            Coupons::create([
                'id' => strtoupper(Str::random(16, 'ABCDEFGHIJKLMNOPQRSTUVWXY1234567890')),
                'prize_id' => $request['prize'],
                'username' => $request['username'],
                'expired_date' => $request['expired']
            ]);
        } catch (\PDOException $e) {
            return $this->handleDatabaseException($e);
        } catch (\Exception $e) {
            return $this->handleDefaultException($e);
        }

        $this->dispatch('success', 'Berhasil membuat kupon');
        $this->reset();
    }

    #[On('show')]
    public function show(Coupons $coupon)
    {
        $this->title = 'Edit Kupon';
        $this->is_edit = true;
        $this->coupon = $coupon;

        $this->username = $coupon->username;
        $this->prize = $coupon->prize_id;
        $this->expired = $coupon->expired_date->format('Y-m-d');
    }

    public function update()
    {
        $request = $this->validate();

        try {
            $coupon = $this->coupon;
            $coupon->username = $request['username'];
            $coupon->prize_id = $request['prize'];
            $coupon->expired_date = $request['expired'];
            $coupon->save();
        } catch (\PDOException $e) {
            return $this->handleDatabaseException($e);
        } catch (\Exception $e) {
            return $this->handleDefaultException($e);
        }

        $this->dispatch('success', 'Kupon berhasil di update');
        $this->reset();
    }

    #[On('delete')]
    public function destroy(Coupons $coupon)
    {
        if ($coupon->id === 'ABCDEFGHIJKLMNOP') {
            return $this->dispatch('swal', message: 'Tidak bisa menghapus kupon untuk testing',  icon: 'error', confirmButtonText: 'Oke');
        }

        try {
            DB::transaction(function () use ($coupon) {
                $id = $coupon->id;
                $isUsed = $coupon->status === self::STATUS_USED;

                $coupon->delete();
                SpinLog::where('coupon_id', $id)->delete();

                if ($isUsed) {
                    unlink(public_path('assets/screenshot/' . $id . '.webp'));
                }
            });
        } catch (\PDOException $e) {
            return $this->handleDatabaseException($e);
        } catch (\Exception $e) {
            return $this->handleDefaultException($e);
        }

        $this->dispatch('success', 'Kupon berhasil dihapus');
    }

    public function hydrate()
    {
        $this->resetErrorBag();
        $this->resetValidation();
    }
}
