<?php

namespace App\Http\Controllers;

use App\Models\Coupons;
use App\Models\SpinLog;
use App\Models\WebConfig;
use App\Models\Wheel;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Intervention\Image\Drivers\Imagick\Driver;
use Intervention\Image\ImageManager;

class MasterController extends Controller
{
    const STATUS_NEW = 0;
    const STATUS_USED = 1;
    const STATUS_EXPIRED = 2;
    const ERROR_MESSAGE_USED = 'Kode kupon sudah terpakai.';
    const ERROR_MESSAGE_EXPIRED = 'Kode kupon kadaluarsa.';

    public function index(Request $request)
    {
        $timestamp = Wheel::first()->updated_at->timestamp;
        $settings = WebConfig::first();
        $history = SpinLog::leftJoin('coupons', 'history.coupon_id', '=', 'coupons.id')
            ->leftJoin('prizes', 'coupons.prize_id', '=', 'prizes.id')
            ->limit(60)
            ->get(['coupons.username', 'prizes.text'])
            ->map(function ($item) {
                $item->username = preg_replace('/(?<=.{3}).(?=.*.{2}$)/', '*', $item->username);
                return $item;
            });

        return view('index', compact('settings', 'history', 'timestamp'));
    }

    public function preload()
    {
        return response()->json([
            'status' => 'success',
            'message' => 'Kode kupon ditemukan',
            'data' => Wheel::first(['claim_number', 'claim_text', 'animation_duration', 'animation_type', 'total_spin', 'segment'])
        ]);
    }

    public function validate_coupon($id, Request $request)
    {
        $wheel = Wheel::first();
        $coupon = Coupons::find($id);

        try {
            if (!$coupon) {
                return response()->json(['status' => 'error', 'message' => 'Kode kupon tidak ditemukan.']);
            }

            if ($coupon->status != self::STATUS_NEW) {
                return response()->json([
                    'status' => 'error',
                    'message' => $coupon->status === self::STATUS_USED ? self::ERROR_MESSAGE_USED : self::ERROR_MESSAGE_EXPIRED
                ]);
            }

            if ($coupon->id !== 'ABCDEFGHIJKLMNOP') {
                if (Carbon::today()->greaterThan($coupon->expired_date)) {
                    $coupon->status = self::STATUS_EXPIRED;
                    $coupon->save();

                    return response()->json(['status' => 'error', 'message' => self::ERROR_MESSAGE_EXPIRED]);
                }

                $coupon->status = self::STATUS_USED;
                $coupon->save();

                $agent = $this->compileUserAgent($request->userAgent());

                SpinLog::create([
                    'coupon_id' => $id,
                    'prize' => $coupon->prize->text,
                    'username' => $coupon->username,
                    'ip_address' => $request->ip(),
                    'browser' => $agent['browser'],
                    'mobile' => $agent['mobile'],
                    'referrer' => $agent['referrer'],
                    'platform' => $agent['platform'],
                ]);
            }
        } catch (\Throwable $th) {
            return response()->json([
                'status' => 'error',
                'message' => 'Terjadi kesalahan internal',
            ]);
        }

        $index = 0;
        foreach ($wheel->segment as $key => $value) {
            if ($value['prize'] === $coupon->prize_id) {
                $index = $key;
                break;
            }
        }

        switch ($index) {
            case 0:
                $angleMin = 0 + 1;
                $angleMax = 45 - 1;
                break;
            case 1:
                $angleMin = 45 + 1;
                $angleMax = 90 - 1;
                break;
            case 2:
                $angleMin = 90 + 1;
                $angleMax = 135 - 1;
                break;
            case 3:
                $angleMin = 135 + 1;
                $angleMax = 180 - 1;
                break;
            case 4:
                $angleMin = 180 + 1;
                $angleMax = 225 - 1;
                break;
            case 5:
                $angleMin = 225 + 1;
                $angleMax = 270 - 1;
                break;
            case 6:
                $angleMin = 270 + 1;
                $angleMax = 315 - 1;
                break;
            case 7:
                $angleMin = 315 + 1;
                $angleMax = 360 - 1;
                break;
        }

        $winAngle = random_int($angleMin, $angleMax);

        return response()->json([
            'status' => 'success',
            'message' => 'Kode kupon ditemukan',
            'data' => [
                'calculation' => $winAngle + 999,
                'username' => $coupon->username
            ]
        ]);
    }

    public function save_image($id, Request $request)
    {
        $formater = new ImageManager(new Driver());
        $image = $formater->read($request->input('image'));

        $webp = $image->toWebp();
        $webp->save(public_path('assets/screenshot/' . $id . '.webp'));

        return response()->json(['status' => 'oke']);
    }

    public function compileUserAgent($agent)
    {
        return [
            'browser' => $this->getBrowser($agent),
            'mobile' => $this->getMobile($agent),
            'referrer' => $this->getReferrer($agent),
            'platform' => $this->getPlatform($agent),
        ];
    }

    /**
     * Get the referrer
     */
    public function getReferrer()
    {
        return empty($_SERVER['HTTP_REFERER']) ? '-' : trim($_SERVER['HTTP_REFERER']);
    }

    /**
     * Set the Platform
     */
    protected function getPlatform($agent)
    {
        if (is_array(config('useragent.platforms')) && config('useragent.platforms')) {
            foreach (config('useragent.platforms') as $key => $val) {
                if (preg_match('|' . preg_quote($key, '|') . '|i', $agent)) {
                    return $val;
                }
            }
        }

        return '-';
    }

    /**
     * Set the Browser
     */
    protected function getBrowser($agent)
    {
        if (is_array(config('useragent.browsers')) && config('useragent.browsers')) {
            foreach (config('useragent.browsers') as $key => $val) {
                if (preg_match('|' . $key . '.*?([0-9\.]+)|i', $agent, $match)) {
                    return $val;
                }
            }
        }

        return '-';
    }

    /**
     * Set the Mobile Device
     */
    protected function getMobile($agent)
    {
        if (is_array(config('useragent.mobiles')) && config('useragent.mobiles')) {
            foreach (config('useragent.mobiles') as $key => $val) {
                if (false !== (stripos($agent, $key))) {
                    return  $val;
                }
            }
        }

        return '-';
    }
}
