<?php

namespace App\Livewire;

use App\Livewire\Traits\ExceptionHandler;
use App\Models\Prizes;
use App\Models\Wheel;
use Illuminate\Support\Arr;
use Intervention\Image\Drivers\Imagick\Driver;
use Intervention\Image\ImageManager;
use Livewire\Attributes\Validate;
use Livewire\Component;
use Livewire\WithFileUploads;

class WheelConfig extends Component
{
    use ExceptionHandler, WithFileUploads;
    /**
     * Give context to custom exception handler
     *
     * @var array $context
     */
    protected $context = 'web setting';

    public $basePrize;

    public $prizes;

    public $prefill = [];

    #[Validate('required', as: 'Durasi animasi', onUpdate: false, translate: false)]
    public $animation_duration;

    #[Validate('required', as: 'Total spin', onUpdate: false, translate: false)]
    public $total_spin;

    #[Validate('nullable|mimes:png,jpg,webp', as: 'Gambar roda putar', onUpdate: false, translate: false)]
    public $wheel_background = null;

    #[Validate('nullable|mimes:png,jpg,webp', as: 'Background website', onUpdate: false, translate: false)]
    public $body_background = null;

    #[Validate('nullable|mimes:png,jpg,webp', as: 'Gambar popup', onUpdate: false, translate: false)]
    public $popup_background = null;

    #[Validate('nullable|mimes:mp3', as: 'Sound opening', onUpdate: false, translate: false)]
    public $opening_sound;

    #[Validate('nullable|mimes:mp3', as: 'Win sound', onUpdate: false, translate: false)]
    public $win_sound = null;

    #[Validate('required', as: 'Segment hadiah', onUpdate: false, translate: false)]
    public $segment;

    public function messages()
    {
        return [
            'required' => ':attribute tidak boleh kosong',
            'string' => ':attribute tidak boleh memiliki karakter lain selain huruf',
            'mimes' => ':attribute hanya menerima ekstensi :values'
        ];
    }

    public function mount()
    {
        $this->basePrize = Prizes::all(['id', 'image', 'text']);
        $this->prizes = $this->basePrize->map(function ($item) {
            return ['text' => $item->id, 'value' => $item->text];
        });

        tap(Wheel::first(), function ($wheel) {
            $this->animation_duration = $wheel->animation_duration;
            $this->total_spin = $wheel->total_spin;

            if ($wheel->segment) {
                $this->prefill = Arr::map($wheel->segment, function ($item) {
                    return ['text' => $item['prize'], 'value' => $item['text']];
                });
            }
        });
    }

    public function update()
    {
        $request = $this->validate();

        try {
            $wheel = Wheel::first();

            $formater = new ImageManager(new Driver());
            if (isset($request['wheel_background'])) {
                $image = $formater->read($request['wheel_background']);
                $webp = $image->toWebp();
                $webp->save(public_path('assets/img/wheel_background.webp'));

                $wheel->wheel_background = 'assets/img/wheel_background.webp';
            }

            if (isset($request['body_background'])) {
                $image = $formater->read($request['body_background']);
                $webp = $image->toWebp();
                $webp->save(public_path('assets/img/background.webp'));

                $wheel->body_background = 'assets/img/background.webp';
            }

            if (isset($request['popup_background'])) {
                $image = $formater->read($request['popup_background']);
                $webp = $image->toWebp();
                $webp->save(public_path('assets/img/popup.webp'));

                $wheel->popup_background = 'assets/img/popup.webp';
            }

            if (isset($request['opening_sound'])) {
                $request['opening_sound']->storeAs('assets/img/', 'opening_sound.mp3', 'real_public');
                $wheel->opening_sound = 'assets/img/opening_sound.mp3';
            }

            if (isset($request['win_sound'])) {
                $request['win_sound']->storeAs('assets/img/', 'win_sound.mp3', 'real_public');
                $wheel->win_sound = 'assets/img/win_sound.mp3';
            }

            $wheel->animation_duration = $request['animation_duration'];
            $wheel->total_spin = $request['total_spin'];
            $wheel->segment = Arr::map($request['segment'], function ($item) {
                $prize = $this->basePrize->where('id', $item['text'])->first();
                return ['prize' => $item['text'], 'image' => 'assets/img/prizes/' . $prize['image'] . '?v=' . now()->timestamp, 'text' => $prize['text']];
            });
            $wheel->save();
        } catch (\PDOException $e) {
            return $this->handleDatabaseException($e);
        } catch (\Exception $e) {
            return $this->handleDefaultException($e);
        }

        $this->dispatch('success', 'Data website berhasil di update');
    }

    public function hydrate()
    {
        $this->resetErrorBag();
        $this->resetValidation();
    }
}
