<?php

namespace App\Livewire\User;

use App\Livewire\Traits\ExceptionHandler;
use App\Models\User;
use Livewire\Component;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;
use Livewire\Attributes\On;
use Livewire\Attributes\Validate;

class UserModal extends Component
{
    use ExceptionHandler;

    /**
     * Give context to custom exception handler
     *
     * @var array $context
     */
    protected $context = 'user';

    public User $user;

    public $title = 'Tambah User';

    public $is_edit = false;

    #[Validate('required|string', as: 'Username', translate: false)]
    public $username;

    #[Validate('required|string', as: 'Nama lengkap', translate: false)]
    public $name;

    #[Validate(as: 'Password', translate: false)]
    public $password;

    public function rules()
    {
        return [
            'password' => [Rule::requiredIf(!$this->is_edit)]
        ];
    }

    public function messages()
    {
        return [
            'required' => ':attribute tidak boleh kosong',
            'string' => ':attribute tidak boleh memiliki karakter lain selain huruf',
        ];
    }

    #[On('create')]
    public function create()
    {
        $this->reset();
    }

    public function store()
    {
        $request = $this->validate();
        $request['password'] = Hash::make($this->password ?: $this->username);

        try {
            User::create($request);
        } catch (\PDOException $e) {
            return $this->handleDatabaseException($e);
        } catch (\Exception $e) {
            return $this->handleDefaultException($e);
        }

        $this->dispatch('success', 'Berhasil membuat user');
        $this->reset();
    }

    #[On('show')]
    public function show(User $user)
    {
        $this->title = 'Edit User';
        $this->is_edit = true;
        $this->user = $user;

        $this->name = $user->name;
        $this->username = $user->username;
    }

    public function update()
    {
        $request = $this->validate();

        try {
            $user = $this->user;
            $user->username = $request['username'];
            $user->name = $request['name'];
            if ($request['password'] !== null) {
                $user->password = Hash::make($request['password']);
            }

            $user->save();
        } catch (\PDOException $e) {
            return $this->handleDatabaseException($e);
        } catch (\Exception $e) {
            return $this->handleDefaultException($e);
        }

        $this->dispatch('success', 'User berhasil di update');
        $this->reset();
    }

    #[On('delete')]
    public function destroy(User $user)
    {
        if (Auth::id() === $user->id) {
            return $this->dispatch('swal', message: 'Tidak bisa menghapus user yang sedang dipakai',  icon: 'error', confirmButtonText: 'Oke');
        }

        try {
            $user->delete();
        } catch (\PDOException $e) {
            return $this->handleDatabaseException($e);
        } catch (\Exception $e) {
            return $this->handleDefaultException($e);
        }

        $this->dispatch('success', 'User berhasil dihapus');
    }

    public function hydrate()
    {
        $this->resetErrorBag();
        $this->resetValidation();
    }
}
